<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;

class Wallet extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        // 'balance',
        'bonus',
        'shares',
    ];
    protected $appends = ['balance'];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function walletCoins()
    {
        // Add a constraint to the relationship to only return WalletCoins
        // where the associated Coin is not soft-deleted.
        return $this->hasMany(WalletCoin::class)->whereHas('coin', function ($query) {
            $query->whereNull('deleted_at');
        });
    }

    public function initializeWalletCoins()
    {
        $coins = Coin::whereNull('deleted_at')->get(); // Only get non-trashed coins

        $existingWalletCoinIds = $this->walletCoins->pluck('coin_id')->toArray(); // Get existing coin IDs in the wallet

        $walletCoinsData = [];

        foreach ($coins as $coin) {
            if (!in_array($coin->id, $existingWalletCoinIds)) { // Check if coin exists for wallet
                $walletCoinsData[] = [
                    'wallet_id' => $this->id, // Use $this->id to access the current wallet's ID
                    'coin_id' => $coin->id,
                    'balance' => 0,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
        }

        if (!empty($walletCoinsData)) {  // Only perform insert if there is data
            WalletCoin::insert($walletCoinsData);
        }
    }

    public function getBalanceAttribute()
    {
        $usdCoin = Coin::where('code', 'USD')->first();

        if ($usdCoin) {
            $walletCoin = $this->walletCoins()->where('coin_id', $usdCoin->id)->first();
            if ($walletCoin) {
                return $walletCoin->balance;
            }
        }

        return 0; // Return 0 if USD coin or wallet coin not found
    }

    public function getTotalBalanceAttribute()
    {
        return $this->walletCoins()
            ->join('coins', 'wallet_coins.coin_id', '=', 'coins.id')
            ->whereNull('coins.deleted_at')
            ->select(DB::raw('COALESCE(SUM(wallet_coins.balance * coins.value), 0) as total_balance'))
            ->value('total_balance');
    }
}
