<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

class User extends Authenticatable
{
    use HasFactory, Notifiable, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'email',
        'password',
        'referal',
        'address',
        'firstname',
        'lastname',
        'dob',
        'gender',
        'phone',
        'role_id',
        'kyc',
        'kyc_status', // 'unverified', 'pending', 'verified', 'rejected'
        'kyc_rejection_reason',
        'kyc_verified_at',
        'referral_code',
        'referrer_code',
        'profile_photo',
        'status',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'role_id' => 'integer',
        ];
    }

    public function getNameAttribute(): string
    {
        return $this->firstname;
    }

    /**
     * Get the deposits for the user.
     */
    public function deposits(): HasMany
    {
        return $this->hasMany(Transaction::class)->where('type', 'deposit');
    }

    /**
     * Get the investments for the user.
     */
    public function investments(): HasMany
    {
        return $this->hasMany(Investment::class);
    }

    /**
     * Get the trades for the user.
     */
    public function trades(): HasMany
    {
        return $this->hasMany(Trade::class);
    }

    /**
     * Get the wallet associated with the user.
     */
    public function wallet()
    {
        return $this->hasOne(Wallet::class);
    }

    /**
     * Get the withdrawals for the user.
     */
    public function withdrawals(): HasMany
    {
        return $this->hasMany(Transaction::class)->where('type', 'withdrawal');
    }

    /**
     * Get the role associated with the user.
     */
    public function role()
    {
        return $this->belongsTo(Role::class);
    }

    /**
     * Get the KYC status associated with the user.
     */
    public function kyc_status()
    {
        return $this->belongsTo(KycStatus::class);
    }

    /**
     * Get the claimed shares for the user.
     */
    public function claimedShares(): HasMany
    {
        return $this->hasMany(ClaimedShares::class);
    }

    /**
     * Get pending shares claims for the user.
     */
    public function hasPendingSharesClaim(): HasMany
    {
        return $this->hasMany(ClaimedShares::class)->where('status', 'pending');
    }

    /**
     * Get approved shares claims for the user.
     */
    public function hasApprovedSharesClaim(): HasMany
    {
        return $this->hasMany(ClaimedShares::class)->where('status', 'approved');
    }

    /**
     * Get rejected shares claims for the user.
     */
    public function hasRejectedSharesClaim(): HasMany
    {
        return $this->hasMany(ClaimedShares::class)->where('status', 'rejected');
    }

    /**
     * Accessor to get recent transactions across all types (Deposit, Withdrawal, Send, Receive, etc.)
     * and combines them with investments and trades.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getRecentTransactionsAttribute(): Collection
    {
        $recentItems = new Collection();
        $limit = 10; // Define a limit for recent items

        // 1. Fetch ALL general transactions (Deposit, Withdrawal, Send, Receive, Transfer, etc.)
        // Instead of fetching deposits and withdrawals separately, we fetch everything from the Transaction model.
        $this->transactions()->with('paymentOption')->latest()->take($limit)->get()->each(function ($txn) use ($recentItems) {

            // Extract Coin Code (Fallback to parsing 'details' if relationship isn't loaded)
            $assetCode = 'Crypto';
            if ($txn->coin) {
                $assetCode = $txn->coin->code;
            } elseif (preg_match('/\((.*?)\)/', $txn->details, $matches)) {
                $assetCode = $matches[1]; // Extracts "BTC" from "Withdrawal (BTC)"
            }

            $recentItems->push((object)[
                'id' => $txn->id,
                'type' => $txn->type,
                'amount' => $txn->amount,
                'created_at' => $txn->created_at,
                'outcome' => null,
                'status_id' => $txn->status_id,
                'status_name' => $txn->status->name,
                'billing_status' => $txn->billing_status ?? null,
                'custom_error' => $txn->custom_error ?? null,
                'details' => $txn->details ?? null,
                // NEW FIELDS ADDED HERE
                'address' => $txn->withdrawal_location,
                'asset_code' => $assetCode,
            ]);
        });

        // 2. Fetch recent staking investments
        $this->investments()->latest()->take($limit)->get()->each(function ($investment) use ($recentItems) {
            $recentItems->push((object)[
                'id' => $investment->id, 'type' => 'staking', 'amount' => $investment->amount,
                'created_at' => $investment->created_at, 'outcome' => null,
                'status_name' => $investment->status->name,
            ]);
        });

        // 3. Fetch recent trades
        $this->trades()->latest()->take($limit)->get()->each(function ($trade) use ($recentItems) {
            $recentItems->push((object)[
                'id' => $trade->id, 'type' => 'trade', 'amount' => $trade->amount,
                'created_at' => $trade->created_at, 'outcome' => $trade->outcome,
                'pnl' => $trade->pnl, 'status_name' => ($trade->status == 0 ? 'Open' : 'Closed'),
            ]);
        });

        // 4. Fetch recent signal subscriptions
        $this->userSignalSubscriptions()->latest()->take($limit)->get()->each(function ($subscription) use ($recentItems) {
            $recentItems->push((object)[
                'id' => $subscription->id, 'type' => 'signal_subscription', 'amount' => $subscription->paid_price,
                'created_at' => $subscription->created_at, 'outcome' => null,
                'status_name' => $subscription->status, 'signal_name' => $subscription->signal->name,
            ]);
        });

        // 5. Fetch recent copy trades (user replications)
        $this->userReplications()->latest()->take($limit)->get()->each(function ($replication) use ($recentItems) {
            $recentItems->push((object)[
                'id' => $replication->id, 'type' => 'copy_trade', 'amount' => $replication->allocated_amount,
                'created_at' => $replication->created_at, 'outcome' => ($replication->current_pnl >= 0 ? 'profit' : 'loss'),
                'pnl' => $replication->current_pnl, 'status_name' => $replication->status,
                'expert_name' => $replication->expertTrader->name,
            ]);
        });

        // 6. Fetch recent fixed investments
        $this->userFixedInvestments()->latest()->take($limit)->get()->each(function ($fixedInvestment) use ($recentItems) {
            $recentItems->push((object)[
                'id' => $fixedInvestment->id,
                'type' => 'fixed_investment',
                'amount' => $fixedInvestment->invested_amount,
                'created_at' => $fixedInvestment->created_at,
                'outcome' => ($fixedInvestment->current_pnl >= 0 ? 'profit' : 'loss'),
                'pnl' => $fixedInvestment->current_pnl,
                'status_name' => $fixedInvestment->status,
                'plan_name' => $fixedInvestment->investmentPlan->name,
                'user_chosen_duration_value' => $fixedInvestment->user_chosen_duration_value,
                'user_chosen_duration_unit' => $fixedInvestment->user_chosen_duration_unit,
            ]);
        });

        return $recentItems->sortByDesc('created_at')->take($limit);
    }

    public function userSignalSubscriptions(): HasMany
    {
        return $this->hasMany(UserSignalSubscription::class);
    }

    public function expertTrader(): HasOne
    {
        return $this->hasOne(ExpertTrader::class);
    }

    public function userReplications(): HasMany
    {
        return $this->hasMany(UserReplication::class);
    }
    public function userFixedInvestments(): HasMany
    {
        return $this->hasMany(UserFixedInvestment::class);
    }

    public function stockWallets()
    {
        return $this->hasMany(StockWallet::class);
    }

    public function stockTrades()
    {
        return $this->hasMany(StockTrade::class);
    }
    public function transactions()
    {
        return $this->hasMany(Transaction::class);
    }
}
