<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules\Password;

class UserProfileController extends Controller
{
    /**
     * Display the user's profile form.
     */
    public function edit()
    {
        return view('user.profile', [
            'user' => Auth::user(),
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(Request $request)
    {
        $user = Auth::user();
            // dd($request->file('photo'));

        $validated = $request->validate([
            'firstname' => ['required', 'string', 'max:255'],
            'lastname'  => ['required', 'string', 'max:255'],
            'email'     => ['required', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'phone'     => ['nullable', 'string', 'max:20'],
            'address'   => ['nullable', 'string', 'max:255'],
            'photo'     => ['nullable', 'image', 'max:2048'], // Max 2MB
        ]);

        // Handle Profile Photo Upload
        if ($request->hasFile('photo')) {
            // Delete old photo if it exists and isn't a default/external URL
            // if ($user->profile_photo && Storage::disk('public')->exists($user->profile_photo)) {
            //     Storage::disk('public')->delete($user->profile_photo);
            // }

            $path = $request->file('photo')->store('profile-photos', 'public');
            $user->profile_photo = $path;

            // If you are using a 'kyc' column for photo as mentioned before, update that instead:
            // $user->kyc = $path;
        }

        $user->fill([
            'firstname' => $validated['firstname'],
            'lastname'  => $validated['lastname'],
            'email'     => $validated['email'],
            'phone'     => $validated['phone'],
            'address'   => $validated['address'],
        ]);

        // Reset email verification if email changes
        if ($user->isDirty('email')) {
            $user->email_verified_at = null;
        }

        $user->save();

        return back()->with('status', 'profile-updated');
    }

    /**
     * Update the user's password.
     */
    public function updatePassword(Request $request)
    {
        $validated = $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', 'confirmed', Password::defaults()],
        ]);

        $request->user()->update([
            'password' => Hash::make($validated['password']),
        ]);

        return back()->with('status', 'password-updated');
    }
}
