<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Wallet;
use App\Models\Transaction;
use App\Models\WalletCoin;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;

class UserController extends Controller
{
    public function generateUniqueTransactionId()
    {
        do {
            $transactionId = strtoupper('TRD' . Str::random(8));
        } while (Transaction::where('transaction_id', $transactionId)->exists());

        return $transactionId;
    }

    public function create()
    {
        $transactionId = strtoupper('TRD' . Str::random(8));

        // Check if the transaction ID already exists in the database
        $depositExists = Deposit::where('transaction_id', $transactionId)->exists();

        while ($depositExists) {
            $transactionId = strtoupper('TRD' . Str::random(8));
            $depositExists = Deposit::where('transaction_id', $transactionId)->exists();
        }

        return view('deposit.create', compact('transactionId'));
    }

    public function store(Request $request)
    {

        $validatedData = $request->validate([
            'amount' => 'required|numeric|min:1',
            'option' => 'required|string',
        ]);
        $paid_transactionId = $request['transaction_id'];

        $deposit = Deposit::create([
            'user_id' => auth()->id(),
            'amount' => $validatedData['amount'],
            'option' => $validatedData['option'],
            'transaction_id' => $request['transaction_id'],
        ]);
        // dd($request['amount']);


        return redirect()->route('deposit.create')->with('success', 'Deposit created successfully!')->with('transaction_id', $paid_transactionId);
    }

    public function admin_view_users()
    {
        // $users = User::all(); // Fetch all users
        $users = User::orderBy('created_at', 'desc')->where('role_id', '!=', 1)->get();

        return view('admin-dash.user-mgt', compact('users'));
    }
    public function admin_view_user_wallets(User $user)
    {
        $wallets = $user->wallet->walletCoins;
        // dd($wallets);
        return view('admin-dash.user-wallets', compact('wallets', 'user'));
    }
    public function admin_view_user_wallet_topup(Request $request, WalletCoin $wallet)
    {
        $validatedData = $request->validate([
            'amount' => 'required|numeric',
        ]);
        // dd($request->amount); // This will show now

        $transactionId = $this->generateUniqueTransactionId();

        $transaction = Transaction::create([
            'user_id' => $wallet->wallet->user_id,
            'amount' => $validatedData['amount'],
            'type' => 'top-up',
            'transaction_id' => $transactionId,
            'payment_option_id' => $wallet->coin->id,//this now serves to identify coin
            // ... other fields based on the transaction type
        ]);
        $wallet->increment('balance', $validatedData['amount']);
        
        if ($validatedData['amount'] > 0) {
            return redirect()->back()->with('success', 'Wallet top-up successfull!');
        }else{
            return redirect()->back()->with('success', 'Wallet decuction successfull!');
        }
    }

    public function admin_update_user(Request $request, $id)
    {
        $validatedData = $request->validate([
            'status' => 'required|string',
            'amount_disbursed' => 'nullable|numeric',
            // ... other validation rules as needed
        ]);

        $deposit = Deposit::findOrFail($id);
        $deposit->update($validatedData);

        return redirect()->route('admin-dash.user-mgt')->with('success', 'KYC updated successfully!');
    }


    public function approve_kyc(Request $request)
    {
        $userId = $request->input('id');
        $user = User::findOrFail($userId);

        $user->update([
            'kyc_status_id' => 2,
        ]);

        return redirect()->back()->with('success', 'KYC approved successfully!');
    }

    public function decline_kyc(Request $request)
    {
        $userId = $request->input('id');
        $user = User::findOrFail($userId);

        $user->update([
            'kyc_status_id' => 3,
        ]);
        return redirect()->back()->with('success_decline', 'KYC declined successfully!');
    }

    public function suspend_user(Request $request)
    {
        $userId = $request->input('id');
        $user = User::findOrFail($userId);

        $user->update([
            'status' => 'suspended',
        ]);
        return redirect()->back()->with('success_n', 'User suspended successfully!');
    }

    public function reactivate_user(Request $request)
    {
        $userId = $request->input('id');
        $user = User::findOrFail($userId);

        $user->update([
            'status' => 'active',
        ]);
        return redirect()->back()->with('success_n', 'User deactivated successfully!');
    }

    public function deactivate_user(Request $request)
    {
        $userId = $request->input('id');
        $user = User::findOrFail($userId);

        $user->update([
            'status' => 'deactivated',
        ]);
        return redirect()->back()->with('success_n', 'User deactivated successfully!');
    }

    public function signInAs(User $user)
    {
        //  Implement the logic to sign in as the selected user.
        //  This might involve using Laravel's auth()->login() method.
        //  You should also consider adding security checks to ensure only authorized users can use this feature.
        Auth::login($user);
        return redirect('/'); // Or wherever you want to redirect the user
    }

    public function getUserData()
    {
        $user = Auth::user();
        $user->load('wallet', 'stockWallets.stock');

        // Eager load stock relationship for claimed shares to avoid N+1 queries
        $stocks = $user->stockWallets->map(function ($stockWallet) {
            return [
                'id' => $stockWallet->stock->id,
                'claimed' => $stockWallet->claimed,
            ];
        });

        return response()->json([
            'wallet' => $user->wallet,
            'stocks' => $stocks,
        ]);
    }

}
