<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Stock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use App\Models\StockWallet;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class StockController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $stocks = Stock::all();
        return view('admin-dash.stocks', compact('stocks'));
    }

    /**
     * Show the form for creating a new resource.
     * Since the form is in the index view as a modal, we don't need a separate create method.
     */
    public function create()
    {
        // Not used as the form is in the index view's modal.
        return redirect()->route('admin.stocks.index');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'value' => 'nullable|numeric|decimal:0,2',
            'ask_value' => 'nullable|numeric|decimal:0,2',
            'ask_shares' => 'nullable|integer|min:0',
            'bid_value' => 'nullable|numeric|decimal:0,2',
            'bid_shares' => 'nullable|integer|min:0',
            'year_range_min' => 'nullable|numeric|decimal:0,2',
            'year_range_max' => 'nullable|numeric|decimal:0,2',
        ]);

        Stock::create($request->all());

        Session::flash('success', 'Stock Added Successfully');
        return redirect()->route('admin.stocks.index');
    }

    /**
     * Display the specified resource.
     * Not directly used in this modal-based approach.
     */
    public function show(Stock $stock)
    {
        return redirect()->route('admin.stocks.index');
    }

    /**
     * Show the form for editing the specified resource.
     * Since the form is in the index view as a modal, we don't need a separate edit method.
     */
    public function edit(Stock $stock)
    {
        // Not used as the form is in the index view's modal.
        return redirect()->route('admin.stocks.index');
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Stock $stock)
    {
        $request->validate([
            'value' => 'nullable|numeric|decimal:0,2',
            'ask_value' => 'nullable|numeric|decimal:0,2',
            'ask_shares' => 'nullable|integer|min:0',
            'bid_value' => 'nullable|numeric|decimal:0,2',
            'bid_shares' => 'nullable|integer|min:0',
            'year_range_min' => 'nullable|numeric|decimal:0,2',
            'year_range_max' => 'nullable|numeric|decimal:0,2',
        ]);

        $stock->update($request->all());

        Session::flash('success', 'Stock Updated Successfully');
        return redirect()->route('admin.stocks.index');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Stock $stock)
    {
        $stock->delete();

        Session::flash('delete_success', 'Stock Deleted Successfully');
        return redirect()->route('admin.stocks.index');
    }

    public function getUserStockAmount(int $stockId)
    {
        // Ensure the user is authenticated
        if (!Auth::check()) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $userId = Auth::id();

        // Find the StockWallet entry for the authenticated user and the given stock ID
        $stockWallet = StockWallet::where('user_id', $userId)
            ->where('stock_id', $stockId)
            ->first();

        if ($stockWallet) {
            return response()->json([
                'stock_id' => $stockId,
                'claimed' => $stockWallet->claimed,
                'unclaimed' => $stockWallet->unclaimed,
                'sold' => $stockWallet->sold,
                'total' => $stockWallet->claimed + $stockWallet->unclaimed,
            ]);
        } else {
            return response()->json([
                'stock_id' => $stockId,
                'claimed' => 0,
                'unclaimed' => 0,
                'sold' => 0,
                'total' => 0,
                'message' => 'No holdings found for this stock.',
            ]);
        }
    }

    public function apiStocks($stockCode): JsonResponse
    {
        $stock = Stock::with('wallet')->where('code', $stockCode)->first();

        if ($stock) {
            return response()->json($stock);
        }

        return response()->json(['error' => 'Stock not found'], 404);
    }
}



