<?php

namespace App\Http\Controllers;

use App\Models\PaymentOption;
use App\Models\Coin;
use App\Models\InvestmentPlan;
use App\Models\Investment;
use App\Models\Wallet;
use App\Models\WalletCoin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class StakeController extends Controller
{
    public function generateUniqueTransactionId()
    {
        do {
            $date = now()->format('dmy'); // Gets current date in DDMMYY format
            $stakeId = strtoupper('STK' . $date . Str::random(8));
        } while (Investment::where('inv_id', $stakeId)->exists());

        return $stakeId;
    }

    public function index()
    {
        $coins = Coin::all();
        $wallet_balance = Auth::user()->wallet->balance;

        return view('stake', compact('coins'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'period' => 'required|int|min:0',
            'amount' => 'required|string|min:0',
            'option' => 'required',
        ]);

        $coin = Coin::find($request->option);
        $wallet = Auth::user()->wallet;
        $walletCoin = WalletCoin::where('wallet_id', $wallet->id)->where('coin_id', $request->option)->first();

        if ($walletCoin->balance >$request->amount ) {
            $investmentId = $this->generateUniqueTransactionId();
            $investment = new Investment([
                'user_id' => Auth::id(),
                'amount' => $request->amount,//i want make usre it is the current price of the investment plan, just incase they can change prices later
                'investment_plan_id' => $coin->id,
                'duration' => $request->period,
                'inv_id' => $investmentId,
                'roi' => $coin->stake_apr
            ]);


            $walletCoin->decrement('balance', $request->amount);
            $investment->save();
            try {
                \Mail::to($investment->user->email)->send(new \App\Mail\InvestmentStartedMail($investment, 'withdrawal'));
            } catch (\Exception $e) {
                Log::error('Error sending TransactionApprovedMail email: ' . $e->getMessage());
            }
        }else{
            return redirect()->back()->with('error', 'Insufficient balance.');
        }
        return redirect()->route('portfolio')->with('success', 'Staked successfully.')->with('investment_id', $investmentId);
    }

    public function admin_view_investments()
    {
        $investments = Investment::orderByDesc('created_at')->get();
        return view('admin-dash.investments', compact('investments'));
    }

    public function admin_update_investment(Request $request)
    {
        $request->validate([
            'id' => 'required',
            'pnl' => 'required|string|in:loss,profit',
            'pnl_amount' => 'required|string',
        ]);

        $id = $request['id'];
        $investment = Investment::findOrFail($id);

        if ($request->input('pnl') == 'loss') {
            $approved_amount = $investment->amount - $request['pnl_amount'];
        } else {
            $approved_amount = $investment->amount + $request['pnl_amount'];
        }

        $investment->pnl_amount = $request['pnl_amount'];
        $investment->disbursed_amount = $approved_amount;
        $investment->status_id = 4;
        $investment->pnl = $request['pnl']==='loss'? '-':'+';
        $investment->disbursed_at = Carbon::now('Africa/Lagos');
        $investment->disbursed_by = auth()->id();

        // Update the wallet balance
        $wallet = Wallet::where('user_id', $investment->user->id)->first();
        $wallet->increment('balance', $approved_amount);

        $investment->save();
        $wallet->save();
        \Mail::to($investment->user->email)->send(new \App\Mail\InvestmentDisbursedMail($investment, 'withdrawal'));

        return redirect()->back()->with('success', 'Trade updated successfully.');
    }


    public function destroy(PaymentOption $paymentOption)
    {

        $paymentOption->delete();

        return redirect()->back()->with('delete_success', 'Payment option deleted successfully.');
    }
}
