<?php

namespace App\Http\Controllers;

use App\Models\Persona;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class PersonaController extends Controller
{
    public function index()
    {
        // Order by active first, then newest
        $personas = Persona::orderBy('is_active', 'desc')->orderBy('created_at', 'desc')->get();
        return view('admin-dash.personas', compact('personas'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'tagline' => 'nullable|string|max:255',
            'primary_color' => 'required|string|max:7',
            'logo' => 'required|image|max:2048',           // Icon
            'logo_with_name' => 'nullable|image|max:2048', // Light Mode
            'logo_dark_full' => 'nullable|image|max:2048', // Dark Mode (NEW)
        ]);

        // 1. Handle Icon
        $logoPath = $request->file('logo')->store('branding', 'public');

        // 2. Handle Light Mode Full
        $logoFullPath = null;
        if ($request->hasFile('logo_with_name')) {
            $logoFullPath = $request->file('logo_with_name')->store('branding', 'public');
        }

        // 3. Handle Dark Mode Full (NEW)
        $logoDarkFullPath = null;
        if ($request->hasFile('logo_dark_full')) {
            $logoDarkFullPath = $request->file('logo_dark_full')->store('branding', 'public');
        }

        Persona::create([
            'name' => $validated['name'],
            'tagline' => $validated['tagline'],
            'primary_color' => $validated['primary_color'],
            'logo_path' => $logoPath,
            'logo_full_path' => $logoFullPath,
            'logo_dark_full_path' => $logoDarkFullPath, // Save to DB
            'is_active' => false,
        ]);

        return back()->with('success', 'New branding persona created successfully.');
    }

    public function update(Request $request, $id)
    {
        $persona = Persona::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'tagline' => 'nullable|string|max:255',
            'primary_color' => 'required|string|max:7',
            'logo' => 'nullable|image|max:2048',
            'logo_with_name' => 'nullable|image|max:2048',
            'logo_dark_full' => 'nullable|image|max:2048', // NEW
        ]);

        $data = [
            'name' => $validated['name'],
            'tagline' => $validated['tagline'],
            'primary_color' => $validated['primary_color'],
        ];

        // 1. Update Icon
        if ($request->hasFile('logo')) {
            if ($persona->logo_path) Storage::disk('public')->delete($persona->logo_path);
            $data['logo_path'] = $request->file('logo')->store('branding', 'public');
        }

        // 2. Update Light Full
        if ($request->hasFile('logo_with_name')) {
            if ($persona->logo_full_path) Storage::disk('public')->delete($persona->logo_full_path);
            $data['logo_full_path'] = $request->file('logo_with_name')->store('branding', 'public');
        }

        // 3. Update Dark Full (NEW)
        if ($request->hasFile('logo_dark_full')) {
            if ($persona->logo_dark_full_path) Storage::disk('public')->delete($persona->logo_dark_full_path);
            $data['logo_dark_full_path'] = $request->file('logo_dark_full')->store('branding', 'public');
        }

        $persona->update($data);

        return back()->with('success', 'Branding updated successfully.');
    }

    public function activate(Persona $persona)
    {
        $persona->update(['is_active' => true]);
        // The Model's boot logic handles deactivating the others
        return redirect()->back()->with('success', $persona->name . ' is now the active persona.');
    }

    public function destroy(Persona $persona)
    {
        if ($persona->is_active) {
            return redirect()->back()->with('error', 'Cannot delete the active persona. Activate another one first.');
        }

        // Optional: Delete files from storage
        // if ($persona->logo_path) Storage::disk('public')->delete($persona->logo_path);

        $persona->delete();
        return redirect()->back()->with('success', 'Persona deleted successfully.');
    }
}
