<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use App\Mail\GenericMail; // Your existing mail class
use Carbon\Carbon;

class KycController extends Controller
{
    // --- USER METHODS ---

    public function index()
    {
        $user = Auth::user();
        return view('user.kyc.index', compact('user'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'document_type' => 'required|string',
            'document_file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:4096',
        ]);

        $user = Auth::user();

        // 1. Upload File
        if ($request->hasFile('document_file')) {
            // Delete old file if exists
            if ($user->kyc) {
                Storage::disk('public')->delete($user->kyc);
            }

            $path = $request->file('document_file')->store('kyc-documents', 'public');

            // 2. Update User
            $user->update([
                'kyc' => $path,
                'kyc_status' => 'pending',
                'kyc_rejection_reason' => null // Clear previous rejection
            ]);
        }

        // 3. Notify Admin (Optional)
        // Mail::to(env('ADMIN_EMAIL'))->send(new GenericMail('New KYC Submission', 'User ' . $user->email . ' has submitted documents.'));

        return redirect()->back()->with('success', 'Documents submitted successfully. Pending review.');
    }

    // --- ADMIN METHODS ---

    public function adminIndex(Request $request)
    {
        $query = User::where('kyc_status', '!=', 'unverified')
                     ->whereNotNull('kyc');

        // Filters
        if ($request->status) {
            $query->where('kyc_status', $request->status);
        } else {
            // Default to pending first, then others
            $query->orderByRaw("FIELD(kyc_status, 'pending', 'verified', 'rejected')");
        }

        $users = $query->latest('updated_at')->paginate(15);

        return view('admin.kyc.index', compact('users'));
    }

    public function adminShow($id)
    {
        $user = User::findOrFail($id);
        return view('admin.kyc.show', compact('user'));
    }

    public function approve($id)
    {
        $user = User::findOrFail($id);

        $user->update([
            'kyc_status' => 'verified',
            'kyc_verified_at' => Carbon::now(),
            'kyc_rejection_reason' => null
        ]);

        // Email User
        try {
            $subject = "Identity Verification Successful";
            $body = "<h2>Congratulations!</h2><p>Your identity has been verified. You now have full access to all features.</p>";
            Mail::to($user->email)->send(new GenericMail($subject, $body));
        } catch (\Exception $e) {}

        return redirect()->route('admin.kyc.index')->with('success', 'User verified successfully.');
    }

    public function reject(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $request->validate(['reason' => 'required|string']);

        $user->update([
            'kyc_status' => 'rejected',
            'kyc_rejection_reason' => $request->reason
        ]);

        // Email User
        try {
            $subject = "Identity Verification Failed";
            $body = "<h2>Action Required</h2><p>Your identity verification failed.</p><p><strong>Reason:</strong> {$request->reason}</p><p>Please re-upload valid documents.</p>";
            Mail::to($user->email)->send(new GenericMail($subject, $body));
        } catch (\Exception $e) {}

        return redirect()->route('admin.kyc.index')->with('success', 'User KYC rejected.');
    }
}
