<?php

namespace App\Http\Controllers;

use App\Models\InvestmentPlan; // For fixed plans
use App\Models\UserFixedInvestment; // For user participation in fixed plans
use App\Models\Investment; // This is now primarily for staking investments
use App\Models\Wallet;
use App\Models\WalletCoin;
use App\Models\Coin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule; // Ensure Rule is imported

class InvestmentController extends Controller
{
    // This generates unique IDs for UserFixedInvestment records now
    public function generateUniqueFixedInvestmentId()
    {
        do {
            $invId = strtoupper('FIXINV' . Str::random(8));
        } while (UserFixedInvestment::where('id', $invId)->exists()); // Assuming 'id' is used for fixed investment IDs

        return $invId;
    }

    // This method is for displaying Fixed Investment Plans
    public function index()
    {
        $investment_plans = InvestmentPlan::all(); // These are the STARTER, SILVER, GOLD plans
        // wallet_balance will still be used to check eligibility
        $payment_options = Coin::all();
        $wallet_balance = Auth::user()->wallet->balance;

        return view('investment', compact('investment_plans', 'wallet_balance', 'payment_options'));
    }

    // This method is for storing a new UserFixedInvestment
    public function store(Request $request)
    {
        $request->validate([
            'investment_plan_id' => 'required|exists:investment_plans,id', // Must be an existing fixed plan
            'invested_amount' => 'required|numeric|min:0.01', // User selected amount within range
            'user_chosen_duration_value' => 'required|integer|min:1', // NEW VALIDATION
            'user_chosen_duration_unit' => ['required', 'string', Rule::in(['days', 'weeks', 'months', 'years'])], // NEW VALIDATION
            'source_wallet_coin_id' => 'required|exists:wallet_coins,id', // Add validation for the selected source wallet
        ]);

        $investment_plan = InvestmentPlan::find($request->investment_plan_id);
        $user = Auth::user();

        // Validate invested_amount against plan's min/max capital
        if ($request->invested_amount < $investment_plan->min_investment || $request->invested_amount > $investment_plan->max_investment) { // Use min/max_investment
            return redirect()->back()->with('error', 'Invested amount must be within the plan\'s specified range: $' . number_format($investment_plan->min_investment) . ' - $' . number_format($investment_plan->max_investment) . '.');
        }

        // Validate user-chosen duration against plan's min/max duration
        if ($request->user_chosen_duration_value < $investment_plan->min_duration_value || $request->user_chosen_duration_value > $investment_plan->max_duration_value) {
            $durrationUnit = $investment_plan->allowed_duration_units[0] ?? 'days';
            return redirect()->back()->with('error', "Chosen duration must be between {$investment_plan->min_duration_value} and {$investment_plan->max_duration_value} {$durrationUnit}.");
        }
        // Validate chosen duration unit is allowed by the plan
        if (!in_array($request->user_chosen_duration_unit, $investment_plan->allowed_duration_units ?? ['days'])) {
            return redirect()->back()->with('error', "Chosen duration unit is not allowed for this plan.");
        }

        // Find the selected wallet coin and its coin value
        $sourceWalletCoin = $user->wallet->walletCoins()->where('id', $request->source_wallet_coin_id)->first();

        // Check if the source wallet coin exists and the balance is sufficient
        if (!$sourceWalletCoin) {
            return redirect()->back()->with('error', 'Invalid source wallet selected.');
        }

        // Calculate the amount to deduct from the selected wallet in its native currency
        $investedAmountInCoin = $request->invested_amount / $sourceWalletCoin->coin->value;

        if ($sourceWalletCoin->balance < $investedAmountInCoin) {
            return redirect()->back()->with('error', 'Insufficient balance in the selected wallet for this investment.');
        }

        DB::beginTransaction();
        try {
            // Deduct invested amount from user's selected wallet
            $sourceWalletCoin->balance -= $investedAmountInCoin;
            // dd($sourceWalletCoin->balance, $investedAmountInCoin, $request->invested_amount, $sourceWalletCoin->coin->value);
            $sourceWalletCoin->save();

            // Calculate end date based on user's chosen duration
            // FIX: Explicitly cast user_chosen_duration_value to int
            $endDate = Carbon::now()->add((int) $request->user_chosen_duration_value, $request->user_chosen_duration_unit);

            // Create UserFixedInvestment record
            $userFixedInvestment = UserFixedInvestment::create([
                'user_id' => $user->id,
                'inv_id' => $this->generateUniqueFixedInvestmentId(),
                'investment_plan_id' => $investment_plan->id,
                'invested_amount' => $request->invested_amount,
                'coin_id' => $sourceWalletCoin->coin->id,
                'current_pnl' => 0, // Starts at 0 P&L
                'start_date' => Carbon::now(),
                'end_date' => $endDate,
                'status' => 'active',
                'user_chosen_duration_value' => $request->user_chosen_duration_value, // STORE NEW
                'user_chosen_duration_unit' => $request->user_chosen_duration_unit,   // STORE NEW
            ]);

            DB::commit();

            // Mail::to($user->email)->send(new \App\Mail\FixedInvestmentStartedMail($userFixedInvestment));

            return redirect()->route('portfolio')->with('success', 'Fixed investment started successfully!')->with('investment_id', $userFixedInvestment->id); // Correct route name

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Fixed investment failed for user ID: {$user->id}. Error: " . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to start investment. Please try again.');
        }
    }


    // --- Existing methods for ADMIN Panel (admin_view_investments, admin_update_investment, destroy) ---
    // These methods currently interact with the 'Investment' model (for staking).
    // They are left unchanged as they apply to staking.

    public function admin_view_investments()
    {
        // This now refers to STAKING investments
        $investments = UserFixedInvestment::with(['user', 'coin'])->latest()->get(); // Updated to statusRel
        // $investments = UserFixedInvestment::with(['user', 'coin', 'statusRel'])->latest()->get(); // Updated to statusRel
        return view('admin-dash.investments', compact('investments'));
    }

    public function admin_update_investment(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:user_fixed_investments,id', // Validating against staking investment ID
            'pnl_type' => 'required|string|in:loss,profit', // Corrected name to pnl_type
            'pnl_amount' => 'required|numeric|min:0', // Corrected name
        ]);

        $id = $request['id'];
        $investment = UserFixedInvestment::findOrFail($id);

        // Calculate disbursed amount
        if ($request->input('pnl_type') == 'loss') { // Corrected name to pnl_type
            $final_disbursed_amount = $investment->invested_amount - $request['pnl_amount'];
        } else { // profit
            $final_disbursed_amount = $investment->invested_amount + $request['pnl_amount'];
        }

        DB::beginTransaction();
        try {
            $investment->pnl_amount = $request['pnl_amount'];
            $investment->disbursed_amount = $final_disbursed_amount;
            $investment->status = UserFixedInvestment::STATUS_DISBURSED; // Use constant
            $investment->pnl = $request['pnl_type']==='loss'? '-':'+'; // Corrected name
            $investment->disbursed_at = Carbon::now('Africa/Lagos');
            $investment->disbursed_by_id = auth()->id();

            // Update the user's wallet coin balance (staking investment returns)
            $user = $investment->user; // Get user from investment
            $wallet = $user->wallet; // Get wallet from user
            $walletCoin = WalletCoin::where('wallet_id', $wallet->id)
                ->where('coin_id', $investment->coin->id) // Use investment->coin->id
                ->first();

            if ($walletCoin) {
                $coin_val = $walletCoin->coin->value;
                $disurse_amt_coin = $final_disbursed_amount / $coin_val;
                // dd($disurse_amt_coin);
                $walletCoin->increment('balance', $disurse_amt_coin);
            } else {
                Log::warning("WalletCoin not found for wallet {$wallet->id} and coin {$investment->coin->id}. Cannot disburse.");
                throw new \Exception("Target WalletCoin not found for disbursement.");
            }

            $investment->save();

            DB::commit();
            // \Mail::to($user->email)->send(new \App\Mail\InvestmentDisbursedMail($investment, 'withdrawal')); // Adjust mail context if needed

            return redirect()->back()->with('success', 'Staking Investment disbursed successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Staking Investment disbursement failed for ID: {$investment->id}. Error: " . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to disburse staking investment: ' . $e->getMessage());
        }
    }

    // This destroy method seems to be for PaymentOption, not Investment
    public function destroy(PaymentOption $paymentOption) // This seems out of place for an InvestmentController
    {
        $paymentOption->delete();
        return redirect()->back()->with('delete_success', 'Payment option deleted successfully.');
    }
}
