<?php

namespace App\Http\Controllers;

use App\Models\ForexPair;
use App\Models\ForexWallet;
use App\Models\ForexTrade;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class ForexController extends Controller
{
    /**
     * API: Get Forex Pair Details
     * Matches the JSON structure expected by your JS
     */
    public function show($id)
    {
        // Frontend might send ID or Symbol
        $pair = ForexPair::where('id', $id)->orWhere('symbol', $id)->firstOrFail();

        // Get User's Wallet
        $wallet = ForexWallet::where('user_id', Auth::id())
                    ->where('forex_pair_id', $pair->id)
                    ->first();

        return response()->json([
            'id' => $pair->id,
            'code' => $pair->symbol,      // JS expects code/symbol
            'name' => $pair->name,
            'market' => 'forex',
            // Map exchange rate to ask/bid for the UI
            'ask_value' => $pair->exchange_rate,
            'bid_value' => $pair->exchange_rate,
            'value' => $pair->exchange_rate,
            'year_range_min' => $pair->year_range_min,
            'year_range_max' => $pair->year_range_max,
            'leverage' => $pair->leverage, // Max leverage
            'wallet' => [
                'balance' => $wallet ? $wallet->balance : 0, // Units held
                'claimed' => $wallet ? $wallet->balance : 0  // JS uses 'claimed' for stocks
            ],
            'claimed' => $wallet ? $wallet->balance : 0
        ]);
    }

    /**
     * API: Execute Forex Trade
     */
    public function store(Request $request)
    {
        $request->validate([
            'stock_id' => 'required|exists:forex_pairs,id', // JS sends 'stock_id'
            'trade_action' => 'required|in:buy,sell',
            'shares_quantity' => 'required|numeric|min:1', // This represents UNITS
            'leverage' => 'required|integer|min:1'
        ]);

        $user = Auth::user();
        $pair = ForexPair::findOrFail($request->stock_id);

        $action = $request->trade_action;
        $units = $request->shares_quantity;
        $leverage = $request->leverage;
        $price = $pair->exchange_rate;

        // Security: Ensure user doesn't exceed max leverage set in DB
        if ($leverage > $pair->leverage) {
            return response()->json(['success' => false, 'message' => "Max leverage allowed is 1:{$pair->leverage}"], 422);
        }

        DB::beginTransaction();
        try {
            $userWallet = $user->wallet; // Your USD Wallet Model

            if ($action === 'buy') {
                // --- BUY LOGIC ---

                // 1. Calculate Required Margin (Cost to User)
                // Formula: (Units * Price) / Leverage
                $marginRequired = ($units * $price) / $leverage;

                if ($userWallet->balance < $marginRequired) {
                    return response()->json(['success' => false, 'message' => 'Insufficient funds for required margin.'], 400);
                }

                // 2. Deduct Margin
                $userWallet->decrement('balance', $marginRequired);

                // 3. Update Forex Wallet
                $fxWallet = ForexWallet::firstOrCreate(
                    ['user_id' => $user->id, 'forex_pair_id' => $pair->id],
                    ['balance' => 0, 'avg_entry_price' => 0, 'avg_leverage' => $leverage]
                );

                // Calculate new Weighted Average Price
                $currentVal = $fxWallet->balance * $fxWallet->avg_entry_price;
                $newVal = $units * $price;
                $newAvgPrice = ($currentVal + $newVal) / ($fxWallet->balance + $units);

                $fxWallet->update([
                    'balance' => $fxWallet->balance + $units,
                    'avg_entry_price' => $newAvgPrice,
                    'avg_leverage' => $leverage // Simplification: updating last used leverage
                ]);

            } else {
                // --- SELL LOGIC ---

                $fxWallet = ForexWallet::where('user_id', $user->id)
                            ->where('forex_pair_id', $pair->id)->first();

                if (!$fxWallet || $fxWallet->balance < $units) {
                    return response()->json(['success' => false, 'message' => 'Insufficient units to sell.'], 400);
                }

                // 1. Calculate Return Amount
                // When selling, user gets back: The Margin they put up +/- Profit/Loss
                // Logic: (Units * CurrentPrice) / LeverageUsed + (Profit/Loss)
                // Simplified Spot Logic: (Units * CurrentPrice) - (BorrowedAmount)

                // Borrowed Amount = (Units * AvgEntry) * (1 - 1/AvgLeverage)
                // To keep it robust for this script, let's use the simple P/L formula:
                // Return = Margin + ((CurrentPrice - EntryPrice) * Units)

                $marginWas = ($units * $fxWallet->avg_entry_price) / $fxWallet->avg_leverage;
                $pnl = ($price - $fxWallet->avg_entry_price) * $units;

                $returnAmount = $marginWas + $pnl;

                // 2. Credit User
                $userWallet->increment('balance', $returnAmount);

                // 3. Decrease Wallet
                $fxWallet->decrement('balance', $units);

                // Optional: Clean up empty wallets
                if ($fxWallet->balance <= 0) {
                    $fxWallet->update(['balance' => 0, 'avg_entry_price' => 0]);
                }
            }

            // 4. Log Trade
            ForexTrade::create([
                'user_id' => $user->id,
                'forex_pair_id' => $pair->id,
                'type' => $action,
                'entry_price' => $price,
                'quantity' => $units,
                'leverage_used' => $leverage,
                'margin_amount' => ($action === 'buy') ? (($units * $price) / $leverage) : 0,
            ]);

            DB::commit();
            return response()->json(['success' => true, 'message' => ucfirst($action) . " order executed successfully."]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }
}
