<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Wallet;
use App\Models\Coin;
use App\Models\WalletCoin;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\VirtualCard;
use Carbon\Carbon;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        $referralCode = request()->query('referral_code');
        return view('auth.register', compact('referralCode'));
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'firstname' => ['required', 'string', 'max:255'],
            'lastname'  => ['required', 'string', 'max:255'],
            'phone'     => ['required', 'string', 'max:20'],
            'address'   => ['required', 'string', 'max:255'],
            'email'     => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:'.User::class],
            'password'  => ['required', 'confirmed', Rules\Password::defaults()],
            'profile_photo' => ['nullable', 'image', 'max:2048'],
        ]);

        // 1. Updated File Storage Logic
        // Using store() automatically handles unique naming (hashing) and placement
        $profilePhotoPath = null;
        if ($request->hasFile('profile_photo')) {
            $profilePhotoPath = $request->file('profile_photo')->store('profile-photos', 'public');
        }

        DB::beginTransaction();

        //

        try {
            $user = User::create([
                'firstname' => $request->firstname,
                'lastname'  => $request->lastname,
                'phone'     => $request->phone,
                'address'   => $request->address,
                'role_id'   => 2,
                'email'     => $request->email,
                'referrer_code' => $request->referrer_code,
                'password'  => Hash::make($request->password),
                // 2. Assign the generated path to the DB column (using 'kyc' per your model schema)
                'kyc'       => $profilePhotoPath,
                'status'    => 'active',
            ]);

            // Create Wallet
            $wallet = Wallet::create([
                'user_id' => $user->id,
                'balance' => 0
            ]);

            // Create Coin Wallets
            $coins = Coin::all();
            $walletCoinsData = [];

            foreach ($coins as $coin) {
                $walletCoinsData[] = [
                    'wallet_id'  => $wallet->id,
                    'coin_id'    => $coin->id,
                    'balance'    => 0,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }

            if (!empty($walletCoinsData)) {
                WalletCoin::insert($walletCoinsData);
            }

            // --- NEW: Generate Virtual Card ---

            // 1. Generate unique 16-digit number (Simple generation)
            // In production, you might want a Luhn-valid generator
            do {
                $cardNumber = '4' . str_pad(mt_rand(1, 999999999999999), 15, '0', STR_PAD_LEFT);
            } while (VirtualCard::where('card_number', $cardNumber)->exists());

            // 2. Generate CVV
            $cvv = str_pad(mt_rand(1, 999), 3, '0', STR_PAD_LEFT);

            // 3. Set Expiry (3 years from now)
            $expiryDate = Carbon::now()->addYears(3);

            VirtualCard::create([
                'user_id'      => $user->id,
                'card_number'  => $cardNumber,
                'cvv'          => $cvv,
                'expiry_month' => $expiryDate->format('m'),
                'expiry_year'  => $expiryDate->format('Y'),
                'status'       => 'inactive', // Default inactive
                'balance'      => 0,
            ]);

            DB::commit();

            event(new Registered($user));

            try {
                Mail::to($user->email)->send(new \App\Mail\RegistrationSuccessMail($user));
            } catch (\Exception $e) {
                Log::error('Error sending registration email: ' . $e->getMessage());
            }

            Auth::login($user);

            if(Auth::user()->role_id === 1 || Auth::user()->role_id === "1"){
                return redirect()->route('admin.users.view');
            } else {
                return redirect()->intended(route('dashboard', absolute: false));
            }

        } catch (\Exception $e) {
            DB::rollBack();
            // If upload happened but DB failed, typically you might want to delete the file here,
            // but for simplicity we allow the transaction rollback to handle data consistency.
            throw $e;
        }
    }
}
