<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use Illuminate\Http\Request;

class TransactionController extends Controller
{
    // List all transactions with filters
    public function index(Request $request)
    {
        $query = Transaction::with(['user', 'status'])
            ->latest();

        // Optional: Filter by Status if you add dropdowns later
        if ($request->has('status') && $request->status != 'all') {
            $query->whereHas('status', function($q) use ($request) {
                $q->where('name', $request->status);
            });
        }

        // Optional: Filter by User or ID
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('transaction_id', 'like', "%$search%")
                  ->orWhereHas('user', function($u) use ($search) {
                      $u->where('firstname', 'like', "%$search%")
                        ->orWhere('lastname', 'like', "%$search%")
                        ->orWhere('email', 'like', "%$search%");
                  });
            });
        }

        $transactions = $query->paginate(15);

        return view('admin.transactions.index', compact('transactions'));
    }

    // ... existing update method ...

    // Single update method for the modal form
    public function update(Request $request, $id)
    {
        $transaction = \App\Models\Transaction::findOrFail($id);

        $request->validate([
            'status_id' => 'required|integer', // 1=Completed, 2=Pending, 3=Rejected
            'billing_status' => 'nullable|in:unpaid,paid,none',
            'custom_error' => 'nullable|string|max:255',
            'amount' => 'required|numeric', // Allow admin to fix amounts
        ]);

        // Update Standard Fields
        $transaction->update([
            'status_id' => $request->status_id,
            'amount' => $request->amount,
            'action_at' => now(),
            'action_by' => auth()->id()
        ]);

        // Update Interruption Logic (Only if provided)
        if ($request->has('billing_status')) {
            $transaction->update([
                'billing_status' => $request->billing_status,
                'custom_error' => $request->custom_error,
            ]);
        }

        return back()->with('success', 'Transaction updated successfully.');
    }
}
