<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\UserFixedInvestment;
use App\Models\Wallet;
use App\Models\WalletCoin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class AdminUserFixedInvestmentController extends Controller
{
    /**
     * Display a listing of user fixed investments.
     */
    public function index()
    {
        // Eager load user and investment plan
        $fixedInvestments = UserFixedInvestment::with(['user', 'investmentPlan'])->latest()->get();
        return view('admin.user-fixed-investments.index', compact('fixedInvestments'));
    }

    /**
     * Disburse a matured user fixed investment.
     */
    public function disburse(Request $request, UserFixedInvestment $userFixedInvestment)
    {
        $request->validate([
            'pnl_type' => 'required|string|in:loss,profit',
            'pnl_amount' => 'required|numeric|min:0',
        ]);

        if ($userFixedInvestment->status !== UserFixedInvestment::STATUS_ACTIVE && $userFixedInvestment->status !== UserFixedInvestment::STATUS_MATURED) {
            return redirect()->back()->with('error', 'Investment is not in a state to be disbursed.');
        }

        DB::beginTransaction();
        try {
            $finalAmount = $userFixedInvestment->invested_amount;
            if ($request->pnl_type === 'profit') {
                $finalAmount += $request->pnl_amount;
            } else { // loss
                $finalAmount -= $request->pnl_amount;
            }

            // Update user's USD wallet (assuming fixed investments are in USD)
            $user = $userFixedInvestment->user;
            $usdWalletCoin = $user->wallet->walletCoins()->whereHas('coin', function($query) {
                $query->where('code', 'USD');
            })->first();

            if (!$usdWalletCoin) {
                Log::error("USD WalletCoin not found for user ID: {$user->id}. Cannot disburse fixed investment.");
                throw new \Exception("User's USD wallet not found.");
            }

            $usdWalletCoin->balance += $finalAmount;
            $usdWalletCoin->save();

            // Update fixed investment record
            $userFixedInvestment->update([
                'current_pnl' => ($request->pnl_type === 'profit' ? $request->pnl_amount : -$request->pnl_amount),
                'status' => UserFixedInvestment::STATUS_DISBURSED,
            ]);

            DB::commit();
            return redirect()->back()->with('success', 'Fixed Investment disbursed successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Fixed Investment disbursement failed for ID: {$userFixedInvestment->id}. Error: " . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to disburse fixed investment: ' . $e->getMessage());
        }
    }

    /**
     * Cancel a user fixed investment.
     */
    public function cancel(UserFixedInvestment $userFixedInvestment)
    {
        if ($userFixedInvestment->status !== UserFixedInvestment::STATUS_ACTIVE && $userFixedInvestment->status !== UserFixedInvestment::STATUS_PENDING) {
            return redirect()->back()->with('error', 'Investment cannot be canceled in its current state.');
        }

        DB::beginTransaction();
        try {
            // Return initial invested amount to user's wallet
            $user = $userFixedInvestment->user;
            $usdWalletCoin = $user->wallet->walletCoins()->whereHas('coin', function($query) {
                $query->where('code', 'USD');
            })->first();

            if (!$usdWalletCoin) {
                Log::error("USD WalletCoin not found for user ID: {$user->id}. Cannot cancel fixed investment and return funds.");
                throw new \Exception("User's USD wallet not found.");
            }

            $usdWalletCoin->balance += $userFixedInvestment->invested_amount;
            $usdWalletCoin->save();

            $userFixedInvestment->update(['status' => UserFixedInvestment::STATUS_CANCELED]);
            DB::commit();
            return redirect()->back()->with('success', 'Fixed Investment canceled successfully and funds returned.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Fixed Investment cancellation failed for ID: {$userFixedInvestment->id}. Error: " . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to cancel fixed investment: ' . $e->getMessage());
        }
    }
}
