<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Stock;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class AdminStockController extends Controller
{
    /**
     * Display a listing of all stocks.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $stocks = Stock::all();
        return view('admin.stocks.index', compact('stocks'));
    }

    /**
     * Store a newly created stock in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:10|unique:stocks,code',
            'symbol' => 'required|string|max:10',
            'market' => ['required', Rule::in(['usd', 'gbx'])],
            'value' => 'required|numeric',
            'ask_value' => 'required|numeric',
            'ask_shares' => 'required|integer',
            'bid_value' => 'required|numeric',
            'bid_shares' => 'required|integer',
            'year_range_min' => 'required|numeric',
            'year_range_max' => 'required|numeric',
            'tradingview_code' => 'nullable|string|max:255',
        ]);

        Stock::create($request->all());

        return back()->with('success', 'Stock added successfully!');
    }

    /**
     * Update the specified stock in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Stock  $stock
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Stock $stock)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => ['required', 'string', 'max:10', Rule::unique('stocks')->ignore($stock->id)],
            'symbol' => 'required|string|max:10',
            'market' => ['required', Rule::in(['usd', 'gbx'])],
            'value' => 'required|numeric',
            'ask_value' => 'required|numeric',
            'ask_shares' => 'required|integer',
            'bid_value' => 'required|numeric',
            'bid_shares' => 'required|integer',
            'year_range_min' => 'required|numeric',
            'year_range_max' => 'required|numeric',
            'tradingview_code' => 'nullable|string|max:255',
        ]);

        $stock->update($request->all());

        return back()->with('success', 'Stock updated successfully!');
    }

    /**
     * Remove the specified stock from storage.
     *
     * @param  \App\Models\Stock  $stock
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Stock $stock)
    {
        $stock->delete();

        return back()->with('success', 'Stock deleted successfully!');
    }
}
